<?php
session_start();

// Database configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'lionnimesha_pattauser');
define('DB_PASS', 'Dk?&EZC]aNK*');
define('DB_NAME', 'lionnimesha_MAGE');

// Dialog API configuration
define('APP_ID', 'APP_066900');
define('APP_KEY', '3caf03ace9e6bca01ff1781e724a563e');
define('PAYMENT_API', 'https://api.dialog.lk/caas/direct/debit');
define('BALANCE_API', 'https://api.dialog.lk/caas/balance/query');
define('SUBSCRIPTION_STATUS', 'https://api.ideamart.io/subscription/getStatus');

// Admin credentials
define('ADMIN_USERNAME', 'pradmp');
define('ADMIN_PASSWORD', '1997');
define('ADMIN_EMAIL', 'pradeepmpofficial@gmail.com');

// CORS headers and content type setting
function setCorsHeaders() {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');
    header('Content-Type: application/json');
    
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit();
    }
}

// Error logging function
function logError($message) {
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message" . PHP_EOL;
    
    if (!is_dir('logs')) {
        mkdir('logs', 0755, true);
    }
    
    file_put_contents('logs/payment_log.txt', $logMessage, FILE_APPEND | LOCK_EX);
}

// Database connection function
function getDbConnection() {
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        logError("Database connection failed: " . $conn->connect_error);
        throw new Exception("Database connection failed");
    }
    return $conn;
}

// API request function
function makeApiRequest($url, $data) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    logError("API Request payload to $url: " . json_encode($data));
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        logError("cURL Error for URL $url: $error");
        return ['httpCode' => 0, 'response' => null, 'error' => $error];
    }

    $responseData = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        logError("JSON decode error for URL $url: " . json_last_error_msg());
        return ['httpCode' => $httpCode, 'response' => null, 'error' => 'Invalid JSON response'];
    }

    logError("API Request to $url - HTTP Code: $httpCode, Response: " . json_encode($responseData));
    return ['httpCode' => $httpCode, 'response' => $responseData, 'error' => null];
}

// Phone number validation function
function validatePhoneNumber($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    if (strlen($phone) === 9 && preg_match('/^7[0-9]{8}$/', $phone)) {
        return 'tel:94' . $phone;
    } elseif (strlen($phone) === 10 && preg_match('/^07[0-9]{8}$/', $phone)) {
        return 'tel:94' . substr($phone, 1);
    } elseif (strlen($phone) === 11 && preg_match('/^947[0-9]{8}$/', $phone)) {
        return 'tel:' . $phone;
    } elseif (strlen($phone) === 12 && preg_match('/^tel:947[0-9]{8}$/', $phone)) {
        return $phone;
    } elseif (strlen($phone) === 13 && preg_match('/^tel:\+947[0-9]{8}$/', $phone)) {
        return str_replace('tel:+', 'tel:', $phone);
    }
    
    logError("Invalid phone number format: $phone");
    return false;
}

// Subscription status check function
function checkSubscriptionStatus($sid) {
    $conn = getDbConnection();
    
    $cleanSid = validatePhoneNumber($sid);
    if (!$cleanSid) {
        $conn->close();
        return ['success' => false, 'message' => 'Invalid phone number format'];
    }

    $query = "SELECT stat, hash FROM subscriptions WHERE pno = ? ORDER BY created_at DESC LIMIT 1";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $cleanSid);
    $stmt->execute();
    $result = $stmt->get_result();
    
    logError("Executing query: $query with phone: $cleanSid");

    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $status = $row['stat'] == 1 ? 'REGISTERED' : 'NOT REGISTERED';
        $hashValue = $row['hash'];
        logError("Subscriber $cleanSid found, status: $status, hash: $hashValue");
        
        $conn->close();
        if ($status === 'REGISTERED') {
            return ['success' => true, 'hash' => $hashValue];
        }
        return ['success' => false, 'message' => 'Subscriber not registered'];
    }

    logError("No subscription found for phone: $cleanSid");
    $conn->close();
    return ['success' => false, 'message' => 'No subscription found'];
}

// Save payment record function
function savePaymentRecord($phoneNumber, $transactionId, $amount, $accountType, $status, $scheduleTime = null) {
    $conn = getDbConnection();
    
    // Get previous payment count
    $query = "SELECT times_paid FROM xpayment WHERE mobile_number = ? ORDER BY timestamp DESC LIMIT 1";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $phoneNumber);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $timesPaid = 1;
    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $timesPaid = $row['times_paid'] + 1;
    }
    
    $sql = "INSERT INTO xpayment (mobile_number, amount, times_paid, timestamp, account_type, transaction_id, status, schedule_time) VALUES (?, ?, ?, NOW(), ?, ?, ?, ?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sdissss", $phoneNumber, $amount, $timesPaid, $accountType, $transactionId, $status, $scheduleTime);
    
    if ($stmt->execute()) {
        $insertId = $conn->insert_id;
        logError("Payment record saved: ID: $insertId, Phone: $phoneNumber, TrxId: $transactionId, Amount: $amount, AccountType: $accountType, Status: $status, ScheduleTime: " . ($scheduleTime ?? 'NULL'));
    } else {
        logError("Error saving payment record: " . $conn->error);
    }
    
    $conn->close();
    return $insertId;
}

// Process single payment function
function processSinglePayment($phoneNumber, $amount = 5.00, $scheduleTime = null) {
    $sid = validatePhoneNumber($phoneNumber);
    if (!$sid) {
        return ['success' => false, 'message' => 'Invalid phone number format'];
    }

    // Check subscription status and fetch hash
    $subscriptionStatus = checkSubscriptionStatus($sid);
    if (!$subscriptionStatus['success']) {
        return $subscriptionStatus;
    }

    $hashValue = $subscriptionStatus['hash'];
    if (!$hashValue) {
        return ['success' => false, 'message' => 'No hash available for subscriber'];
    }

    // If scheduled, save as pending and return
    if ($scheduleTime) {
        $transactionId = 'SCHED_' . crc32(uniqid());
        $insertId = savePaymentRecord($sid, $transactionId, $amount, 'UNKNOWN', 'pending', $scheduleTime);
        return ['success' => true, 'message' => 'Payment scheduled successfully', 'paymentId' => $insertId];
    }

    // Check balance
    $balanceRequest = [
        'applicationId' => APP_ID,
        'password' => APP_KEY,
        'subscriberId' => $hashValue,
        'paymentInstrumentName' => 'Mobile Account'
    ];

    logError("Sending balance request for subscriber: $sid, hash: $hashValue");
    $balanceResponse = makeApiRequest(BALANCE_API, $balanceRequest);

    if ($balanceResponse['error'] || $balanceResponse['httpCode'] !== 200) {
        $errorMsg = $balanceResponse['error'] ?? ($balanceResponse['response']['statusDetail'] ?? 'Balance check failed');
        logError("Balance check failed for subscriber: $sid, Error: $errorMsg");
        return ['success' => false, 'message' => "Balance check failed: $errorMsg"];
    }

    $balanceData = $balanceResponse['response'];
    $accountType = $balanceData['accountType'] ?? '';
    $availableBalance = floatval($balanceData['chargeableBalance'] ?? 0);

    $isActive = false;
    if ($accountType === 'POSTPAID') {
        $isActive = true;
        logError("Subscriber $sid is POSTPAID, considered active");
    } elseif ($accountType === 'PREPAID' && $availableBalance >= $amount) {
        $isActive = true;
        logError("Subscriber $sid is PREPAID with sufficient balance: $availableBalance");
    } else {
        logError("Subscriber $sid is $accountType with insufficient balance: $availableBalance");
        return ['success' => false, 'message' => 'Account not active or insufficient balance'];
    }

    if (!$isActive) {
        return ['success' => false, 'message' => 'Account not active'];
    }

    // Process payment
    $externalTrxId = crc32(uniqid());
    $paymentRequest = [
        'applicationId' => APP_ID,
        'password' => APP_KEY,
        'subscriberId' => $hashValue,
        'paymentInstrumentName' => 'Mobile Account',
        'amount' => number_format($amount, 2, '.', ''),
        'currency' => 'LKR',
        'externalTrxId' => $externalTrxId
    ];

    logError("Sending payment request for subscriber: $sid, TrxId: $externalTrxId, hash: $hashValue");
    $paymentResponse = makeApiRequest(PAYMENT_API, $paymentRequest);

    $status = ($paymentResponse['error'] || $paymentResponse['httpCode'] !== 200 || 
        ($paymentResponse['response']['statusCode'] ?? '') !== 'S1000') ? 'failed' : 'success';

    if ($status === 'failed') {
        $errorMsg = $paymentResponse['error'] ?? ($paymentResponse['response']['statusDetail'] ?? 'Payment failed');
        logError("Payment failed for subscriber: $sid, TrxId: $externalTrxId, Error: $errorMsg");
        savePaymentRecord($sid, $externalTrxId, $amount, $accountType, $status);
        return ['success' => false, 'message' => "Payment failed: $errorMsg"];
    }

    // Save payment record
    savePaymentRecord($sid, $externalTrxId, $amount, $accountType, $status);

    logError("Payment successful for subscriber: $sid, TrxId: $externalTrxId");
    return ['success' => true, 'message' => 'Payment successful. Access granted!'];
}

// Process pending scheduled payments
function processPendingPayments() {
    $conn = getDbConnection();
    
    $query = "SELECT id, mobile_number, amount FROM xpayment WHERE status = 'pending' AND schedule_time <= NOW()";
    $result = $conn->query($query);
    
    $results = [];
    $processed = 0;
    while ($row = $result->fetch_assoc()) {
        $paymentId = $row['id'];
        $phoneNumber = $row['mobile_number'];
        $amount = floatval($row['amount']);
        
        $paymentResult = processSinglePayment($phoneNumber, $amount);
        
        // Update payment status
        $updateQuery = "UPDATE xpayment SET status = ?, transaction_id = ? WHERE id = ?";
        $stmt = $conn->prepare($updateQuery);
        $newStatus = $paymentResult['success'] ? 'success' : 'failed';
        $transactionId = $paymentResult['success'] ? crc32(uniqid()) : ('FAILED_' . crc32(uniqid()));
        $stmt->bind_param("ssi", $newStatus, $transactionId, $paymentId);
        $stmt->execute();
        
        $results[$paymentId] = [
            'phoneNumber' => $phoneNumber,
            'result' => $paymentResult
        ];
        $processed++;
    }
    
    $conn->close();
    return ['success' => true, 'processed' => $processed, 'results' => $results];
}

// Cancel scheduled payment
function cancelScheduledPayment($paymentId) {
    $conn = getDbConnection();
    
    $query = "UPDATE xpayment SET status = 'cancelled' WHERE id = ? AND status = 'pending'";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $paymentId);
    
    if ($stmt->execute() && $stmt->affected_rows > 0) {
        logError("Payment cancelled: ID: $paymentId");
        $conn->close();
        return ['success' => true, 'message' => 'Payment cancelled successfully'];
    }
    
    logError("Failed to cancel payment: ID: $paymentId");
    $conn->close();
    return ['success' => false, 'message' => 'Payment not found or already processed'];
}

// Cancel all scheduled bulk payments
function cancelBulkPayments() {
    $conn = getDbConnection();
    
    $query = "UPDATE xpayment SET status = 'cancelled' WHERE status = 'pending' AND transaction_id LIKE 'SCHED_%'";
    $result = $conn->query($query);
    
    $affectedRows = $conn->affected_rows;
    logError("Cancelled $affectedRows bulk payments");
    
    $conn->close();
    return ['success' => true, 'message' => "Cancelled $affectedRows scheduled payments"];
}

// Payment history function
function getPaymentHistory($limit = 50) {
    $conn = getDbConnection();
    
    $query = "SELECT id, mobile_number, amount, times_paid, timestamp, account_type, transaction_id, status, schedule_time FROM xpayment ORDER BY timestamp DESC LIMIT ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $limit);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $payments = [];
    while ($row = $result->fetch_assoc()) {
        $payments[] = [
            'id' => $row['id'],
            'phoneNumber' => $row['mobile_number'],
            'amount' => floatval($row['amount']),
            'timesPaid' => $row['times_paid'],
            'timestamp' => $row['timestamp'],
            'accountType' => $row['account_type'],
            'transactionId' => $row['transaction_id'],
            'status' => $row['status'],
            'scheduleTime' => $row['schedule_time']
        ];
    }
    
    $conn->close();
    return $payments;
}

// Dashboard statistics function
function getDashboardStats() {
    $conn = getDbConnection();
    
    $query = "SELECT COUNT(*) as count, SUM(amount) as total FROM xpayment WHERE DATE(timestamp) = CURDATE() AND status = 'success'";
    $result = $conn->query($query);
    $todayStats = $result->fetch_assoc();
    
    $query = "SELECT COUNT(*) as total FROM xpayment WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND status = 'success'";
    $result = $conn->query($query);
    $totalPayments = $result->fetch_assoc()['total'];
    
    $query = "SELECT COUNT(*) as success FROM xpayment WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND status = 'success'";
    $result = $conn->query($query);
    $successPayments = $result->fetch_assoc()['success'];
    
    $query = "SELECT COUNT(*) as pending FROM xpayment WHERE status = 'pending'";
    $result = $conn->query($query);
    $pendingPayments = $result->fetch_assoc()['pending'];
    
    $successRate = $totalPayments > 0 ? ($successPayments / $totalPayments) * 100 : 0;
    
    $conn->close();
    
    return [
        'todayPayments' => intval($todayStats['count']),
        'todayRevenue' => floatval($todayStats['total'] ?? 0),
        'successRate' => round($successRate, 1),
        'totalPayments' => $totalPayments,
        'pendingPayments' => $pendingPayments
    ];
}

// Fetch subscriber numbers from subscriptions table, excluding duplicates for the day
function getSubscriberNumbers() {
    $conn = getDbConnection();
    
    $query = "SELECT pno FROM subscriptions WHERE stat = 1 
              AND pno COLLATE utf8mb4_unicode_ci NOT IN (
                  SELECT mobile_number FROM xpayment 
                  WHERE DATE(timestamp) = CURDATE() 
                  AND status = 'success'
              ) ORDER BY created_at DESC";
    $result = $conn->query($query);
    
    $numbers = [];
    while ($row = $result->fetch_assoc()) {
        $numbers[] = $row['pno'];
    }
    
    $conn->close();
    return $numbers;
}

// Authentication functions
function adminLogin($username, $password) {
    if ($username === ADMIN_USERNAME && $password === ADMIN_PASSWORD) {
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_username'] = $username;
        $_SESSION['login_time'] = time();
        
        logError("Admin login successful: $username");
        return ['success' => true, 'message' => 'Login successful'];
    }
    
    logError("Failed login attempt: $username");
    return ['success' => false, 'message' => 'Invalid credentials'];
}

function adminLogout() {
    session_destroy();
    return ['success' => true, 'message' => 'Logged out successfully'];
}

function isLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function sendPasswordReset($email) {
    if ($email !== ADMIN_EMAIL) {
        return ['success' => false, 'message' => 'Email not found'];
    }
    
    $token = bin2hex(random_bytes(32));
    $resetLink = "https://mage.lk/Dashboard/reset-password.php?token=$token";
    
    logError("Password reset requested for: $email, Reset link: $resetLink");
    
    return ['success' => true, 'message' => 'Password reset link sent to your email'];
}

function requireAuth() {
    if (!isLoggedIn()) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Authentication required']);
        exit;
    }
}

// API request handling
$isApiRequest = strpos($_SERVER['REQUEST_URI'], '/api/') !== false || 
                isset($_GET['api']) || 
                ($_SERVER['REQUEST_METHOD'] === 'POST' && 
                strpos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false);

if ($isApiRequest) {
    setCorsHeaders();
    
    $method = $_SERVER['REQUEST_METHOD'];
    $requestUri = $_SERVER['REQUEST_URI'];
    
    logError("Request Method: $method");
    logError("Request URI: $requestUri");
    logError("Raw Input: " . file_get_contents('php://input'));
    logError("GET Parameters: " . json_encode($_GET));
    logError("POST Parameters: " . json_encode($_POST));
    
    $pathParts = explode('/', trim(parse_url($requestUri, PHP_URL_PATH), '/'));
    $action = end($pathParts);
    
    try {
        // Process pending payments on every API request
        $pendingResult = processPendingPayments();
        
        if ($method === 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid JSON input: ' . json_last_error_msg());
            }
            
            $action = $input['action'] ?? $action;
            logError("POST Action determined: $action");
            
            switch ($action) {
                case 'login':
                    $username = $input['username'] ?? '';
                    $password = $input['password'] ?? '';
                    $result = adminLogin($username, $password);
                    break;
                    
                case 'logout':
                    $result = adminLogout();
                    break;
                    
                case 'reset-password':
                    $email = $input['email'] ?? '';
                    $result = sendPasswordReset($email);
                    break;
                
                case 'single-payment':
                    requireAuth();
                    $phoneNumber = $input['pno'] ?? '';
                    $amount = floatval($input['amount'] ?? 5.00);
                    $scheduleTime = $input['scheduleTime'] ?? null;
                    
                    if (empty($phoneNumber)) {
                        throw new Exception('Phone number is required');
                    }
                    
                    $result = processSinglePayment($phoneNumber, $amount, $scheduleTime);
                    break;
                    
                case 'bulk-payment':
                    requireAuth();
                    $amount = floatval($input['amount'] ?? 5.00);
                    $scheduleTime = $input['scheduleTime'] ?? null;
                    
                    $numbers = getSubscriberNumbers();
                    if (empty($numbers)) {
                        throw new Exception('No eligible subscribers found for today');
                    }
                    
                    $results = [];
                    $successful = 0;
                    $failed = 0;
                    $paymentIds = [];
                    
                    if ($scheduleTime) {
                        foreach ($numbers as $number) {
                            if (empty(trim($number))) continue;
                            
                            $paymentResult = processSinglePayment(trim($number), $amount, $scheduleTime);
                            $results[$number] = $paymentResult;
                            
                            if ($paymentResult['success']) {
                                $successful++;
                                $paymentIds[] = $paymentResult['paymentId'];
                            } else {
                                $failed++;
                            }
                        }
                    } else {
                        foreach ($numbers as $number) {
                            if (empty(trim($number))) continue;
                            
                            $paymentResult = processSinglePayment(trim($number), $amount);
                            $results[$number] = $paymentResult;
                            
                            if ($paymentResult['success']) {
                                $successful++;
                            } else {
                                $failed++;
                            }
                            
                            usleep(500000); // 0.5 seconds delay
                        }
                    }
                    
                    $result = [
                        'success' => true,
                        'message' => "Bulk payment completed. Success: $successful, Failed: $failed",
                        'results' => $results,
                        'summary' => [
                            'successful' => $successful,
                            'failed' => $failed,
                            'total' => count($numbers)
                        ],
                        'paymentIds' => $paymentIds
                    ];
                    break;
                    
                case 'cancel-payment':
                    requireAuth();
                    $paymentId = $input['paymentId'] ?? null;
                    if (!$paymentId) {
                        throw new Exception('Payment ID is required');
                    }
                    $result = cancelScheduledPayment($paymentId);
                    break;
                    
                case 'cancel-bulk':
                    requireAuth();
                    $result = cancelBulkPayments();
                    break;
                    
                case 'check-pending':
                    requireAuth();
                    $result = $pendingResult;
                    break;
                    
                default:
                    throw new Exception("Invalid action: $action");
            }
            
        } elseif ($method === 'GET') {
            $action = $_GET['action'] ?? $action;
            logError("GET Action determined: $action");
            
            switch ($action) {
                case 'check-auth':
                    $result = [
                        'success' => true,
                        'authenticated' => isLoggedIn()
                    ];
                    break;
                    
                case 'payment-history':
                    requireAuth();
                    $limit = intval($_GET['limit'] ?? 50);
                    $result = [
                        'success' => true,
                        'data' => getPaymentHistory($limit)
                    ];
                    break;
                    
                case 'dashboard-stats':
                    requireAuth();
                    $result = [
                        'success' => true,
                        'data' => getDashboardStats()
                    ];
                    break;
                    
                default:
                    throw new Exception("Invalid GET action: $action");
            }
            
        } else {
            http_response_code(405);
            throw new Exception('Method not allowed');
        }
        
    } catch (Exception $e) {
        $result = ['success' => false, 'message' => $e->getMessage()];
        logError("Error: " . $e->getMessage());
    }
    
    echo json_encode($result);
    exit;
}

// HTML Dashboard Interface
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .loading { animation: spin 1s linear infinite; }
        @keyframes spin { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }
        .fade-in { animation: fadeIn 0.3s ease-in; }
        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
        .slide-up { animation: slideUp 0.3s ease-out; }
        @keyframes slideUp { from { transform: translateY(20px); opacity: 0; } to { transform: translateY(0); opacity: 1; } }
    </style>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <!-- Login Form -->
    <div id="loginForm" class="min-h-screen flex items-center justify-center p-4">
        <div class="bg-white rounded-2xl shadow-2xl p-8 w-full max-w-md slide-up">
            <div class="text-center mb-8">
                <div class="mx-auto w-20 h-20 bg-gradient-to-r from-blue-600 to-indigo-600 rounded-full flex items-center justify-center mb-4">
                    <i class="fas fa-credit-card text-white text-2xl"></i>
                </div>
                <h1 class="text-3xl font-bold text-gray-800 mb-2">Dashboard</h1>
                <p class="text-gray-600">Log Now</p>
            </div>

            <div id="loginSection">
                <form id="loginFormElement" class="space-y-6">
                    <div class="relative">
                        <i class="fas fa-user absolute left-3 top-4 text-gray-400"></i>
                        <input type="text" id="username" placeholder="Username" 
                               class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all" required>
                    </div>
                    <div class="relative">
                        <i class="fas fa-lock absolute left-3 top-4 text-gray-400"></i>
                        <input type="password" id="password" placeholder="Password" 
                               class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all" required>
                    </div>
                    <button type="submit" class="w-full bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white font-medium py-3 px-4 rounded-lg transition-all duration-200 flex items-center justify-center space-x-2">
                        <span>Login</span>
                        <i class="fas fa-arrow-right"></i>
                    </button>
                </form>
                <button onclick="showPasswordReset()" class="w-full text-blue-600 hover:text-blue-700 font-medium py-2 mt-4 transition-colors">
                    Forgot Password?
                </button>
            </div>

            <div id="resetSection" class="hidden">
                <form id="resetFormElement" class="space-y-4">
                    <div class="relative">
                        <i class="fas fa-envelope absolute left-3 top-4 text-gray-400"></i>
                        <input type="email" id="resetEmail" placeholder="pradeepmpofficial@gmail.com" 
                               class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                    </div>
                    <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-3 px-4 rounded-lg transition-colors">
                        Send Reset Link
                    </button>
                </form>
                <button onclick="showLogin()" class="w-full text-gray-600 hover:text-gray-700 font-medium py-2 mt-4 transition-colors">
                    Back to Login
                </button>
            </div>

            <div id="resetSuccess" class="hidden text-center space-y-4">
                <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto">
                    <i class="fas fa-check text-green-600 text-2xl"></i>
                </div>
                <h3 class="text-lg font-medium text-gray-800">Reset Link Sent!</h3>
                <p class="text-gray-600">Check your email for password reset instructions.</p>
                <button onclick="showLogin()" class="text-blue-600 hover:text-blue-700 font-medium transition-colors">
                    Back to Login
                </button>
            </div>
        </div>
    </div>

    <!-- Dashboard -->
    <div id="dashboard" class="hidden min-h-screen">
        <!-- Header -->
        <header class="bg-white shadow-lg border-b">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="flex justify-between items-center h-16">
                    <div class="flex items-center space-x-4">
                        <div class="w-10 h-10 bg-gradient-to-r from-blue-600 to-indigo-600 rounded-lg flex items-center justify-center">
                            <i class="fas fa-credit-card text-white"></i>
                        </div>
                        <div>
                            <h1 class="text-xl font-bold text-gray-900">Dashboard</h1>
                            <p class="text-sm text-gray-500">Admin</p>
                        </div>
                    </div>
                    <button onclick="logout()" class="flex items-center space-x-2 text-gray-700 hover:text-red-600 transition-colors">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Logout</span>
                    </button>
                </div>
            </div>
        </header>

        <!-- Navigation -->
        <nav class="bg-white shadow-sm">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="flex space-x-8">
                    <button onclick="showTab('single')" class="tab-btn flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm transition-colors active">
                        <i class="fas fa-credit-card"></i>
                        <span>Single Payment</span>
                    </button>
                    <button onclick="showTab('bulk')" class="tab-btn flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm transition-colors">
                        <i class="fas fa-upload"></i>
                        <span>Bulk Payment</span>
                    </button>
                    <button onclick="showTab('history')" class="tab-btn flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm transition-colors">
                        <i class="fas fa-history"></i>
                        <span>Payment History</span>
                    </button>
                    <button onclick="showTab('stats')" class="tab-btn flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm transition-colors">
                        <i class="fas fa-chart-line"></i>
                        <span>Statistics</span>
                    </button>
                </div>
            </div>
        </nav>

        <!-- Main Content -->
        <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            <!-- Single Payment Tab -->
            <div id="singleTab" class="tab-content">
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h2 class="text-xl font-semibold text-gray-900 mb-6 flex items-center">
                        <i class="fas fa-credit-card mr-2 text-blue-600"></i>
                        Single Payment Processing
                    </h2>
                    
                    <form id="singlePaymentForm" class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number</label>
                                <div class="relative">
                                    <i class="fas fa-phone absolute left-3 top-3 text-gray-400"></i>
                                    <input type="tel" id="singlePhone" placeholder="0771234567" 
                                           class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                </div>
                                <p class="mt-1 text-sm text-gray-500">Enter a Dialog phone number</p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Amount (LKR)</label>
                                <div class="relative">
                                    <i class="fas fa-dollar-sign absolute left-3 top-3 text-gray-400"></i>
                                    <input type="number" id="singleAmount" value="5.00" step="0.01" min="0.01" 
                                           class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                </div>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Schedule Time (Optional)</label>
                                <div class="relative">
                                    <i class="fas fa-calendar absolute left-3 top-3 text-gray-400"></i>
                                    <input type="datetime-local" id="singleScheduleTime" 
                                           class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                </div>
                                <p class="mt-1 text-sm text-gray-500">Leave blank for immediate payment</p>
                            </div>
                        </div>
                        
                        <div class="flex space-x-4">
                            <button type="submit" id="singleStartBtn" class="flex-1 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white font-medium py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2">
                                <i class="fas fa-play"></i>
                                <span>Start Payment</span>
                            </button>
                            <button type="button" id="singleStopBtn" onclick="stopSinglePayment()" class="flex-1 bg-red-600 hover:bg-red-700 text-white font-medium py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2 hidden">
                                <i class="fas fa-stop"></i>
                                <span>Stop Payment</span>
                            </button>
                        </div>
                    </form>
                    
                    <div id="singleResult" class="mt-6 hidden"></div>
                </div>
            </div>

            <!-- Bulk Payment Tab -->
            <div id="bulkTab" class="tab-content hidden">
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h2 class="text-xl font-semibold text-gray-900 mb-6 flex items-center">
                        <i class="fas fa-upload mr-2 text-green-600"></i>
                        Bulk Payment Processing
                    </h2>
                    
                    <div class="space-y-6">
                        <!-- Amount Input -->
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Amount per Payment (LKR)</label>
                                <div class="relative">
                                    <i class="fas fa-dollar-sign absolute left-3 top-3 text-gray-400"></i>
                                    <input type="number" id="bulkAmount" value="5.00" step="0.01" min="0.01" 
                                           class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                </div>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Schedule Time (Optional)</label>
                                <div class="relative">
                                    <i class="fas fa-calendar absolute left-3 top-3 text-gray-400"></i>
                                    <input type="datetime-local" id="bulkScheduleTime" 
                                           class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                </div>
                                <p class="mt-1 text-sm text-gray-500">Leave blank for immediate payment</p>
                            </div>
                        </div>

                        <!-- Start/Stop Buttons -->
                        <div class="flex space-x-4">
                            <button id="bulkStartBtn" onclick="startBulkPayment()" class="flex-1 bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-700 text-white font-medium py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2">
                                <i class="fas fa-play"></i>
                                <span>Start Bulk Payment</span>
                            </button>
                            <button id="bulkStopBtn" onclick="stopBulkPayment()" class="flex-1 bg-red-600 hover:bg-red-700 text-white font-medium py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2 hidden">
                                <i class="fas fa-stop"></i>
                                <span>Stop Bulk Payment</span>
                            </button>
                        </div>

                        <!-- Results Summary -->
                        <div id="bulkResults" class="hidden border rounded-lg p-4 bg-gray-50">
                            <h3 class="font-medium text-gray-900 mb-2">Payment Results</h3>
                            <div class="grid grid-cols-2 gap-4 text-sm">
                                <div class="flex items-center space-x-2">
                                    <i class="fas fa-check-circle text-green-600"></i>
                                    <span id="successCount">Successful: 0</span>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <i class="fas fa-times-circle text-red-600"></i>
                                    <span id="failedCount">Failed: 0</span>
                                </div>
                            </div>
                            <div id="bulkResultDetails" class="mt-4 text-sm max-h-96 overflow-y-auto"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Payment History Tab -->
            <div id="historyTab" class="tab-content hidden">
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h2 class="text-xl font-semibold text-gray-900 mb-6 flex items-center">
                        <i class="fas fa-history mr-2 text-purple-600"></i>
                        Payment History
                    </h2>
                    
                    <div id="historyContent" class="overflow-x-auto">
                        <div class="text-center py-8">
                            <i class="fas fa-spinner fa-spin text-4xl text-gray-400 mb-4"></i>
                            <p class="text-gray-500">Loading payment history...</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Statistics Tab -->
            <div id="statsTab" class="tab-content hidden">
                <div class="space-y-6">
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6" id="statsCards">
                        <div class="text-center py-8">
                            <i class="fas fa-spinner fa-spin text-4xl text-gray-400 mb-4"></i>
                            <p class="text-gray-500">Loading statistics...</p>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        let isLoggedIn = false;
        let currentTab = 'single';
        let singlePaymentId = null;
        let bulkPaymentIds = [];

        // Check authentication status and process pending payments on page load
        document.addEventListener('DOMContentLoaded', function() {
            checkAuthStatus();
            // Start interval to check for pending payments every 30 seconds
            setInterval(checkPendingPayments, 30000);
        });

        async function checkAuthStatus() {
            try {
                const response = await fetch('?action=check-auth');
                const data = await response.json();
                
                if (data.success && data.authenticated) {
                    showDashboard();
                } else {
                    showLoginForm();
                }
            } catch (error) {
                console.error('Auth check failed:', error);
                showLoginForm();
            }
        }

        async function checkPendingPayments() {
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'check-pending' })
                });
                const data = await response.json();
                if (data.success && data.processed > 0) {
                    showAlert(`Processed ${data.processed} pending payments`, 'success');
                    if (currentTab === 'history') loadPaymentHistory();
                    if (currentTab === 'stats') loadDashboardStats();
                    
                    // Reset single payment UI if payment was processed
                    if (singlePaymentId && data.results[singlePaymentId]) {
                        document.getElementById('singleStopBtn').classList.add('hidden');
                        document.getElementById('singleStartBtn').classList.remove('hidden');
                        singlePaymentId = null;
                    }
                }
            } catch (error) {
                console.error('Pending payments check failed:', error);
            }
        }

        // Login form handling
        document.getElementById('loginFormElement').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const username = document.getElementById('username').value;
            const password = document.getElementById('password').value;
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'login', username, password })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showDashboard();
                } else {
                    showAlert(data.message, 'error');
                }
            } catch (error) {
                showAlert('Login failed. Please try again.', 'error');
            }
        });

        // Password reset form handling
        document.getElementById('resetFormElement').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const email = document.getElementById('resetEmail').value;
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'reset-password', email })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    document.getElementById('resetSection').classList.add('hidden');
                    document.getElementById('resetSuccess').classList.remove('hidden');
                } else {
                    showAlert(data.message, 'error');
                }
            } catch (error) {
                showAlert('Reset request failed. Please try again.', 'error');
            }
        });

        // Single payment form handling
        document.getElementById('singlePaymentForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const phoneNumber = document.getElementById('singlePhone').value;
            const amount = parseFloat(document.getElementById('singleAmount').value);
            const scheduleTime = document.getElementById('singleScheduleTime').value;
            
            const submitBtn = document.getElementById('singleStartBtn');
            const stopBtn = document.getElementById('singleStopBtn');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Processing...';
            submitBtn.disabled = true;
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                        action: 'single-payment', 
                        pno: phoneNumber, 
                        amount: amount,
                        scheduleTime: scheduleTime || null
                    })
                });
                
                const data = await response.json();
                showPaymentResult('singleResult', data);
                
                if (data.success && scheduleTime) {
                    singlePaymentId = data.paymentId;
                    stopBtn.classList.remove('hidden');
                    submitBtn.classList.add('hidden');
                }
                
            } catch (error) {
                showPaymentResult('singleResult', { 
                    success: false, 
                    message: 'Network error occurred. Please try again.' 
                });
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        });

        async function stopSinglePayment() {
            if (!singlePaymentId) {
                showAlert('No scheduled payment to stop', 'error');
                return;
            }
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                        action: 'cancel-payment', 
                        paymentId: singlePaymentId
                    })
                });
                
                const data = await response.json();
                showPaymentResult('singleResult', data);
                
                // Always hide stop button after attempting to cancel
                document.getElementById('singleStopBtn').classList.add('hidden');
                document.getElementById('singleStartBtn').classList.remove('hidden');
                singlePaymentId = null;
                
            } catch (error) {
                showAlert('Failed to stop payment. Please try again.', 'error');
                document.getElementById('singleStopBtn').classList.add('hidden');
                document.getElementById('singleStartBtn').classList.remove('hidden');
                singlePaymentId = null;
            }
        }

        async function startBulkPayment() {
            const amount = parseFloat(document.getElementById('bulkAmount').value);
            const scheduleTime = document.getElementById('bulkScheduleTime').value;
            
            const startBtn = document.getElementById('bulkStartBtn');
            const stopBtn = document.getElementById('bulkStopBtn');
            const originalText = startBtn.innerHTML;
            startBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Processing...';
            startBtn.disabled = true;
            
            document.getElementById('bulkResults').classList.add('hidden');
            document.getElementById('bulkResultDetails').innerHTML = '';
            bulkPaymentIds = [];
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                        action: 'bulk-payment', 
                        amount: amount,
                        scheduleTime: scheduleTime || null
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    const results = data.results;
                    let successful = data.summary.successful;
                    let failed = data.summary.failed;
                    
                    // Store payment IDs for scheduled payments
                    if (scheduleTime) {
                        bulkPaymentIds = data.paymentIds || [];
                    }
                    
                    // Display detailed results
                    const detailsContainer = document.getElementById('bulkResultDetails');
                    for (const [number, result] of Object.entries(results)) {
                        const resultDiv = document.createElement('div');
                        resultDiv.className = `flex items-center space-x-2 p-2 ${result.success ? 'bg-green-50' : 'bg-red-50'}`;
                        resultDiv.innerHTML = `
                            <i class="fas ${result.success ? 'fa-check-circle text-green-600' : 'fa-times-circle text-red-600'}"></i>
                            <span>${number}: ${result.message}</span>
                        `;
                        detailsContainer.appendChild(resultDiv);
                    }
                    
                    document.getElementById('successCount').textContent = `Successful: ${successful}`;
                    document.getElementById('failedCount').textContent = `Failed: ${failed}`;
                    document.getElementById('bulkResults').classList.remove('hidden');
                    
                    showAlert(`Bulk payment completed. Successful: ${successful}, Failed: ${failed}`, 'info');
                    
                    if (scheduleTime && bulkPaymentIds.length > 0) {
                        stopBtn.classList.remove('hidden');
                        startBtn.classList.add('hidden');
                    }
                } else {
                    showAlert(data.message, 'error');
                }
                
            } catch (error) {
                showAlert('Bulk payment failed. Please try again.', 'error');
            } finally {
                startBtn.innerHTML = originalText;
                startBtn.disabled = false;
            }
        }

        async function stopBulkPayment() {
            if (bulkPaymentIds.length === 0) {
                showAlert('No scheduled bulk payments to stop', 'error');
                return;
            }
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'cancel-bulk' })
                });
                
                const data = await response.json();
                showAlert(data.message, data.success ? 'success' : 'error');
                
                if (data.success) {
                    document.getElementById('bulkStopBtn').classList.add('hidden');
                    document.getElementById('bulkStartBtn').classList.remove('hidden');
                    document.getElementById('bulkResults').classList.add('hidden');
                    bulkPaymentIds = [];
                }
                
            } catch (error) {
                showAlert('Failed to stop bulk payment. Please try again.', 'error');
            }
        }

        function showLoginForm() {
            document.getElementById('loginForm').classList.remove('hidden');
            document.getElementById('dashboard').classList.add('hidden');
            isLoggedIn = false;
        }

        function showDashboard() {
            document.getElementById('loginForm').classList.add('hidden');
            document.getElementById('dashboard').classList.remove('hidden');
            isLoggedIn = true;
            
            loadPaymentHistory();
            loadDashboardStats();
            checkPendingPayments();
        }

        function showPasswordReset() {
            document.getElementById('loginSection').classList.add('hidden');
            document.getElementById('resetSection').classList.remove('hidden');
        }

        function showLogin() {
            document.getElementById('resetSection').classList.add('hidden');
            document.getElementById('resetSuccess').classList.add('hidden');
            document.getElementById('loginSection').classList.remove('hidden');
        }

        async function logout() {
            try {
                await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'logout' })
                });
            } catch (error) {
                console.error('Logout error:', error);
            }
            
            showLoginForm();
        }

        function showTab(tabName) {
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.add('hidden');
            });
            
            document.getElementById(tabName + 'Tab').classList.remove('hidden');
            
            document.querySelectorAll('.tab-btn').forEach(btn => {
                btn.classList.remove('active', 'border-blue-500', 'text-blue-600');
                btn.classList.add('border-transparent', 'text-gray-500');
            });
            
            event.target.classList.add('active', 'border-blue-500', 'text-blue-600');
            event.target.classList.remove('border-transparent', 'text-gray-500');
            
            currentTab = tabName;
            
            if (tabName === 'history') {
                loadPaymentHistory();
            } else if (tabName === 'stats') {
                loadDashboardStats();
            }
            checkPendingPayments();
        }

        async function loadPaymentHistory() {
            const container = document.getElementById('historyContent');
            
            try {
                const response = await fetch('?action=payment-history&limit=50');
                const data = await response.json();
                
                if (data.success && data.data.length > 0) {
                    container.innerHTML = `
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Phone Number</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Times Paid</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Account Type</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Transaction ID</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Timestamp</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Schedule Time</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                ${data.data.map(payment => `
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <i class="fas fa-phone text-gray-400 mr-2"></i>
                                                <span class="text-sm font-medium text-gray-900">${payment.phoneNumber}</span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <i class="fas fa-dollar-sign text-gray-400 mr-2"></i>
                                                <span class="text-sm text-gray-900">LKR ${payment.amount.toFixed(2)}</span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${payment.timesPaid}</td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${payment.accountType || 'N/A'}</td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusClasses(payment.status)}">
                                                ${getStatusIcon(payment.status)}
                                                <span class="ml-1 capitalize">${payment.status}</span>
                                            </span>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${payment.transactionId || 'N/A'}</td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <i class="fas fa-clock text-gray-400 mr-2"></i>
                                                <span class="text-sm text-gray-900">${new Date(payment.timestamp).toLocaleString()}</span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <i class="fas fa-calendar text-gray-400 mr-2"></i>
                                                <span class="text-sm text-gray-900">${payment.scheduleTime ? new Date(payment.scheduleTime).toLocaleString() : 'N/A'}</span>
                                            </div>
                                        </td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    `;
                } else {
                    container.innerHTML = `
                        <div class="text-center py-12">
                            <i class="fas fa-history text-6xl text-gray-400 mb-4"></i>
                            <p class="text-gray-500">No payment history found</p>
                        </div>
                    `;
                }
            } catch (error) {
                container.innerHTML = `
                    <div class="text-center py-12">
                        <i class="fas fa-exclamation-triangle text-6xl text-red-400 mb-4"></i>
                        <p class="text-red-500">Failed to load payment history</p>
                    </div>
                `;
            }
        }

        async function loadDashboardStats() {
            const container = document.getElementById('statsCards');
            
            try {
                const response = await fetch('?action=dashboard-stats');
                const data = await response.json();
                
                if (data.success) {
                    const stats = data.data;
                    container.innerHTML = `
                        <div class="bg-white rounded-xl shadow-lg p-6">
                            <div class="flex items-center justify-between">
                                <div class="p-3 rounded-lg bg-blue-50">
                                    <i class="fas fa-dollar-sign text-2xl text-blue-600"></i>
                                </div>
                                <div class="text-sm font-medium text-green-600">+12%</div>
                            </div>
                            <div class="mt-4">
                                <h3 class="text-2xl font-bold text-gray-900">${stats.totalPayments}</h3>
                                <p class="text-sm text-gray-500">Total Payments</p>
                            </div>
                        </div>
                        
                        <div class="bg-white rounded-xl shadow-lg p-6">
                            <div class="flex items-center justify-between">
                                <div class="p-3 rounded-lg bg-green-50">
                                    <i class="fas fa-check-circle text-2xl text-green-600"></i>
                                </div>
                                <div class="text-sm font-medium text-green-600">${stats.successRate}%</div>
                            </div>
                            <div class="mt-4">
                                <h3 class="text-2xl font-bold text-gray-900">${stats.successRate}%</h3>
                                <p class="text-sm text-gray-500">Success Rate</p>
                            </div>
                        </div>
                        
                        <div class="bg-white rounded-xl shadow-lg p-6">
                            <div class="flex items-center justify-between">
                                <div class="p-3 rounded-lg bg-purple-50">
                                    <i class="fas fa-calendar-day text-2xl text-purple-600"></i>
                                </div>
                                <div class="text-sm font-medium text-green-600">Today</div>
                            </div>
                            <div class="mt-4">
                                <h3 class="text-2xl font-bold text-gray-900">${stats.todayPayments}</h3>
                                <p class="text-sm text-gray-500">Today's Payments</p>
                            </div>
                        </div>
                        
                        <div class="bg-white rounded-xl shadow-lg p-6">
                            <div class="flex items-center justify-between">
                                <div class="p-3 rounded-lg bg-yellow-50">
                                    <i class="fas fa-coins text-2xl text-yellow-600"></i>
                                </div>
                                <div class="text-sm font-medium text-green-600">LKR</div>
                            </div>
                            <div class="mt-4">
                                <h3 class="text-2xl font-bold text-gray-900">${stats.todayRevenue.toFixed(2)}</h3>
                                <p class="text-sm text-gray-500">Today's Revenue</p>
                            </div>
                        </div>
                        
                        <div class="bg-white rounded-xl shadow-lg p-6">
                            <div class="flex items-center justify-between">
                                <div class="p-3 rounded-lg bg-orange-50">
                                    <i class="fas fa-clock text-2xl text-orange-600"></i>
                                </div>
                                <div class="text-sm font-medium text-orange-600">Pending</div>
                            </div>
                            <div class="mt-4">
                                <h3 class="text-2xl font-bold text-gray-900">${stats.pendingPayments}</h3>
                                <p class="text-sm text-gray-500">Pending Payments</p>
                            </div>
                        </div>
                    `;
                }
            } catch (error) {
                container.innerHTML = `
                    <div class="col-span-4 text-center py-12">
                        <i class="fas fa-exclamation-triangle text-6xl text-red-400 mb-4"></i>
                        <p class="text-red-500">Failed to load statistics</p>
                    </div>
                `;
            }
        }

        function showPaymentResult(containerId, result) {
            const container = document.getElementById(containerId);
            const alertClass = result.success ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200';
            const iconClass = result.success ? 'fas fa-check-circle text-green-600' : 'fas fa-times-circle text-red-600';
            const textClass = result.success ? 'text-green-800' : 'text-red-800';
            const titleClass = result.success ? 'text-green-800' : 'text-red-800';
            
            container.innerHTML = `
                <div class="${alertClass} border rounded-lg p-4 fade-in">
                    <div class="flex items-center space-x-2">
                        <i class="${iconClass}"></i>
                        <span class="font-medium ${titleClass}">
                            ${result.success ? 'Payment Successful!' : 'Payment Failed'}
                        </span>
                    </div>
                    <p class="mt-2 ${textClass}">${result.message}</p>
                </div>
            `;
            container.classList.remove('hidden');
        }

        function showAlert(message, type = 'info') {
            const alertColors = {
                success: 'bg-green-100 border-green-400 text-green-700',
                error: 'bg-red-100 border-red-400 text-red-700',
                info: 'bg-blue-100 border-blue-400 text-blue-700',
                warning: 'bg-yellow-100 border-yellow-400 text-yellow-700'
            };
            
            const alert = document.createElement('div');
            alert.className = `fixed top-4 right-4 ${alertColors[type]} border px-4 py-3 rounded z-50 fade-in`;
            alert.innerHTML = `
                <div class="flex items-center justify-between">
                    <span>${message}</span>
                    <button onclick="this.parentElement.parentElement.remove()" class="ml-4">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `;
            
            document.body.appendChild(alert);
            
            setTimeout(() => {
                if (alert.parentElement) {
                    alert.remove();
                }
            }, 5000);
        }

        function getStatusClasses(status) {
            switch (status) {
                case 'success':
                    return 'bg-green-100 text-green-800';
                case 'failed':
                    return 'bg-red-100 text-red-800';
                case 'pending':
                    return 'bg-yellow-100 text-yellow-800';
                case 'cancelled':
                    return 'bg-gray-100 text-gray-800';
                default:
                    return 'bg-gray-100 text-gray-800';
            }
        }

        function getStatusIcon(status) {
            switch (status) {
                case 'success':
                    return '<i class="fas fa-check-circle"></i>';
                case 'failed':
                    return '<i class="fas fa-times-circle"></i>';
                case 'pending':
                    return '<i class="fas fa-clock"></i>';
                case 'cancelled':
                    return '<i class="fas fa-ban"></i>';
                default:
                    return '<i class="fas fa-question-circle"></i>';
            }
        }

        // Initialize first tab as active
        document.addEventListener('DOMContentLoaded', function() {
            const firstTab = document.querySelector('.tab-btn');
            if (firstTab) {
                firstTab.classList.add('active', 'border-blue-500', 'text-blue-600');
                firstTab.classList.remove('border-transparent', 'text-gray-500');
            }
        });
    </script>
</body>
</html>