<?php
class PaymentProcessor {
    private $conn;
    
    public function __construct() {
        $this->conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
        if ($this->conn->connect_error) {
            throw new Exception("Database connection failed: " . $this->conn->connect_error);
        }
    }
    
    public function __destruct() {
        if ($this->conn) {
            $this->conn->close();
        }
    }
    
    // API request function
    private function makeApiRequest($url, $data) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        
        logError("API Request payload to $url: " . json_encode($data));
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            logError("cURL Error for URL $url: $error");
            return ['httpCode' => 0, 'response' => null, 'error' => $error];
        }

        $responseData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            logError("JSON decode error for URL $url: " . json_last_error_msg());
            return ['httpCode' => $httpCode, 'response' => null, 'error' => 'Invalid JSON response'];
        }

        logError("API Request to $url - HTTP Code: $httpCode, Response: " . json_encode($responseData));
        return ['httpCode' => $httpCode, 'response' => $responseData, 'error' => null];
    }
    
    // Phone number validation
    public function validatePhoneNumber($phone) {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (strlen($phone) === 9 && preg_match('/^7[0-9]{8}$/', $phone)) {
            return 'tel:94' . $phone;
        } elseif (strlen($phone) === 10 && preg_match('/^07[0-9]{8}$/', $phone)) {
            return 'tel:94' . substr($phone, 1);
        } elseif (strlen($phone) === 11 && preg_match('/^947[0-9]{8}$/', $phone)) {
            return 'tel:' . $phone;
        } elseif (strlen($phone) === 12 && preg_match('/^tel:947[0-9]{8}$/', $phone)) {
            return $phone;
        } elseif (strlen($phone) === 13 && preg_match('/^tel:\+947[0-9]{8}$/', $phone)) {
            return str_replace('tel:+', 'tel:', $phone);
        }
        
        logError("Invalid phone number format: $phone");
        return false;
    }
    
    // Check subscription status
    private function checkSubscriptionStatus($sid, $providedHash = null) {
        $cleanSid = $this->validatePhoneNumber($sid);
        if (!$cleanSid) {
            return ['success' => false, 'message' => 'Invalid phone number format'];
        }

        $dbPhone = $cleanSid;
        $dbHash = $providedHash;

        $query = "SELECT stat, hash FROM subscriptions WHERE pno = ?";
        if ($dbHash) {
            $query .= " AND hash = ?";
        }
        $query .= " ORDER BY created_at DESC LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        if ($dbHash) {
            $stmt->bind_param("ss", $dbPhone, $dbHash);
        } else {
            $stmt->bind_param("s", $dbPhone);
        }
        $stmt->execute();
        $result = $stmt->get_result();
        
        logError("Executing query: $query with phone: $dbPhone" . ($dbHash ? ", hash: $dbHash" : ""));

        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $status = $row['stat'] == 1 ? 'REGISTERED' : 'NOT REGISTERED';
            $hashValue = $row['hash'];
            logError("Subscriber $cleanSid found, status: $status, hash: $hashValue");
            
            if ($status === 'REGISTERED') {
                return ['success' => true, 'hash' => $hashValue];
            }
            return ['success' => false, 'message' => 'Subscriber not registered'];
        }

        logError("No subscription found for phone: $cleanSid");
        return ['success' => false, 'message' => 'No subscription found'];
    }
    
    // Check payment validity
    private function checkPaymentValidity($sid) {
        $cleanSid = $this->validatePhoneNumber($sid);
        if (!$cleanSid) {
            return false;
        }

        $query = "SELECT expires_at, times_paid, amount FROM payments WHERE mobile_number = ? AND expires_at > NOW() ORDER BY timestamp DESC LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("s", $cleanSid);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            logError("Valid payment found for subscriber: $cleanSid, expires at: " . $row['expires_at']);
            return true;
        }

        return false;
    }
    
    // Process single payment
    public function processSinglePayment($phoneNumber, $amount = 5.00, $hash = null) {
        $sid = $this->validatePhoneNumber($phoneNumber);
        if (!$sid) {
            return ['success' => false, 'message' => 'Invalid phone number format'];
        }

        // Check subscription status
        $subscriptionStatus = $this->checkSubscriptionStatus($sid, $hash);
        if (!$subscriptionStatus['success']) {
            return $subscriptionStatus;
        }

        $hashValue = $subscriptionStatus['hash'];
        if (!$hashValue) {
            return ['success' => false, 'message' => 'No hash available for subscriber'];
        }

        // Check existing payment
        if ($this->checkPaymentValidity($sid)) {
            return ['success' => true, 'message' => 'Existing payment still valid. Access granted for 48 hours!'];
        }

        // Check balance
        $balanceRequest = [
            'applicationId' => APP_ID,
            'password' => APP_KEY,
            'subscriberId' => $hashValue,
            'paymentInstrumentName' => 'Mobile Account'
        ];

        $balanceResponse = $this->makeApiRequest(BALANCE_API, $balanceRequest);

        if ($balanceResponse['error'] || $balanceResponse['httpCode'] !== 200) {
            $errorMsg = $balanceResponse['error'] ?? 'Balance check failed';
            return ['success' => false, 'message' => "Balance check failed: $errorMsg"];
        }

        $balanceData = $balanceResponse['response'];
        $accountType = $balanceData['accountType'] ?? '';
        $availableBalance = floatval($balanceData['chargeableBalance'] ?? 0);

        // Check account status
        $isActive = false;
        if ($accountType === 'POSTPAID') {
            $isActive = true;
        } elseif ($accountType === 'PREPAID' && $availableBalance >= $amount) {
            $isActive = true;
        }

        if (!$isActive) {
            return ['success' => false, 'message' => 'Account not active or insufficient balance'];
        }

        // Process payment
        $externalTrxId = crc32(uniqid());
        $paymentRequest = [
            'applicationId' => APP_ID,
            'password' => APP_KEY,
            'subscriberId' => $hashValue,
            'paymentInstrumentName' => 'Mobile Account',
            'amount' => number_format($amount, 2, '.', ''),
            'currency' => 'LKR',
            'externalTrxId' => $externalTrxId
        ];

        $paymentResponse = $this->makeApiRequest(PAYMENT_API, $paymentRequest);

        if ($paymentResponse['error'] || $paymentResponse['httpCode'] !== 200 || 
            ($paymentResponse['response']['statusCode'] ?? '') !== 'S1000') {
            $errorMsg = $paymentResponse['error'] ?? ($paymentResponse['response']['statusDetail'] ?? 'Payment failed');
            return ['success' => false, 'message' => "Payment failed: $errorMsg"];
        }

        // Save payment record
        $this->savePaymentRecord($sid, $externalTrxId, $amount);

        return ['success' => true, 'message' => 'Payment successful. Access granted for 48 hours!'];
    }
    
    // Save payment record
    private function savePaymentRecord($phoneNumber, $transactionId, $amount) {
        // Get previous payment count
        $query = "SELECT times_paid FROM payments WHERE mobile_number = ? ORDER BY timestamp DESC LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("s", $phoneNumber);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $timesPaid = 1;
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $timesPaid = $row['times_paid'] + 1;
        }

        $expiresAt = date('Y-m-d H:i:s', strtotime('+48 hours'));
        
        $sql = "INSERT INTO payments (transaction_id, mobile_number, timestamp, expires_at, times_paid, amount) VALUES (?, ?, NOW(), ?, ?, ?)";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("sssid", $transactionId, $phoneNumber, $expiresAt, $timesPaid, $amount);
        
        if ($stmt->execute()) {
            logError("Payment record saved: Phone: $phoneNumber, TrxId: $transactionId, Amount: $amount");
        } else {
            logError("Error saving payment record: " . $this->conn->error);
        }
    }
    
    // Get payment history
    public function getPaymentHistory($limit = 50) {
        $query = "SELECT * FROM payments ORDER BY timestamp DESC LIMIT ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $payments = [];
        while ($row = $result->fetch_assoc()) {
            $payments[] = [
                'id' => $row['id'],
                'phoneNumber' => $row['mobile_number'],
                'amount' => floatval($row['amount']),
                'status' => (strtotime($row['expires_at']) > time()) ? 'success' : 'expired',
                'timestamp' => $row['timestamp'],
                'transactionId' => $row['transaction_id'],
                'timesPaid' => $row['times_paid']
            ];
        }
        
        return $payments;
    }
    
    // Get dashboard statistics
    public function getDashboardStats() {
        $stats = [];
        
        // Total payments today
        $query = "SELECT COUNT(*) as count, SUM(amount) as total FROM payments WHERE DATE(timestamp) = CURDATE()";
        $result = $this->conn->query($query);
        $todayStats = $result->fetch_assoc();
        
        // Success rate (last 30 days)
        $query = "SELECT COUNT(*) as total FROM payments WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        $result = $this->conn->query($query);
        $totalPayments = $result->fetch_assoc()['total'];
        
        $query = "SELECT COUNT(*) as success FROM payments WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND expires_at > timestamp";
        $result = $this->conn->query($query);
        $successPayments = $result->fetch_assoc()['success'];
        
        $successRate = $totalPayments > 0 ? ($successPayments / $totalPayments) * 100 : 0;
        
        return [
            'todayPayments' => intval($todayStats['count']),
            'todayRevenue' => floatval($todayStats['total']),
            'successRate' => round($successRate, 1),
            'totalPayments' => $totalPayments
        ];
    }
}
?>