<?php
require_once 'config.php';
require_once 'classes/PaymentProcessor.php';
require_once 'classes/AuthManager.php';

setCorsHeaders();

$auth = new AuthManager();
$auth->requireAuth(); // Require authentication for payment operations

$processor = new PaymentProcessor();
$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? 'single';
    
    try {
        switch ($action) {
            case 'single':
                $phoneNumber = $input['pno'] ?? '';
                $amount = floatval($input['amount'] ?? 5.00);
                $hash = $input['hash'] ?? null;
                
                if (empty($phoneNumber)) {
                    throw new Exception('Phone number is required');
                }
                
                $result = $processor->processSinglePayment($phoneNumber, $amount, $hash);
                break;
                
            case 'bulk':
                $numbers = $input['numbers'] ?? [];
                $amount = floatval($input['amount'] ?? 5.00);
                
                if (empty($numbers)) {
                    throw new Exception('Phone numbers are required');
                }
                
                $results = [];
                $successful = 0;
                $failed = 0;
                
                foreach ($numbers as $number) {
                    if (empty(trim($number))) continue;
                    
                    $result = $processor->processSinglePayment(trim($number), $amount);
                    $results[$number] = $result['success'];
                    
                    if ($result['success']) {
                        $successful++;
                    } else {
                        $failed++;
                    }
                    
                    // Add small delay between requests
                    usleep(500000); // 0.5 seconds
                }
                
                $result = [
                    'success' => true,
                    'message' => "Bulk payment completed. Success: $successful, Failed: $failed",
                    'results' => $results,
                    'summary' => [
                        'successful' => $successful,
                        'failed' => $failed,
                        'total' => count($numbers)
                    ]
                ];
                break;
                
            default:
                throw new Exception('Invalid action');
        }
        
    } catch (Exception $e) {
        $result = ['success' => false, 'message' => $e->getMessage()];
        logError("Payment processing error: " . $e->getMessage());
    }
    
    echo json_encode($result);
    
} elseif ($method === 'GET') {
    // Handle GET requests for payment history or stats
    $action = $_GET['action'] ?? 'history';
    
    try {
        switch ($action) {
            case 'history':
                $limit = intval($_GET['limit'] ?? 50);
                $result = [
                    'success' => true,
                    'data' => $processor->getPaymentHistory($limit)
                ];
                break;
                
            case 'stats':
                $result = [
                    'success' => true,
                    'data' => $processor->getDashboardStats()
                ];
                break;
                
            default:
                throw new Exception('Invalid action');
        }
        
    } catch (Exception $e) {
        $result = ['success' => false, 'message' => $e->getMessage()];
    }
    
    echo json_encode($result);
    
} else {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}
?>